import { redirect } from 'next/navigation';
import { getCurrentUser } from '@/modules/auth/actions';
import { ApprovalService } from '@/modules/admin/approvals/service';
import { MediatorReviewActions } from '@/components/admin/MediatorReviewActions';

const approvalService = new ApprovalService();

export default async function MediatorReviewPage({
  params,
}: {
  params: { id: string };
}) {
  const user = await getCurrentUser();

  if (!user || user.role !== 'ADMIN') {
    redirect('/login');
  }

  const mediator = await approvalService.getMediatorForReview(params.id);

  return (
    <div>
      <div className="mb-8">
        <a href="/admin/approvals" className="text-sm text-primary-600 hover:text-primary-700 mb-2 inline-block">
          ← Back to approvals
        </a>
        <h1 className="text-3xl font-bold text-gray-900">
          Review Mediator Application
        </h1>
        <p className="text-gray-600 mt-2">
          {mediator.firstName} {mediator.lastName}
        </p>
      </div>

      <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
        {/* Main Content */}
        <div className="lg:col-span-2 space-y-6">
          {/* Basic Info */}
          <div className="card">
            <h2 className="text-xl font-semibold mb-4">Basic Information</h2>
            
            <dl className="grid grid-cols-1 gap-4">
              <div>
                <dt className="text-sm font-medium text-gray-500">Name</dt>
                <dd className="mt-1 text-sm text-gray-900">
                  {mediator.displayName || `${mediator.firstName} ${mediator.lastName}`}
                </dd>
              </div>

              <div>
                <dt className="text-sm font-medium text-gray-500">Email</dt>
                <dd className="mt-1 text-sm text-gray-900">{mediator.user.email}</dd>
              </div>

              {mediator.phone && (
                <div>
                  <dt className="text-sm font-medium text-gray-500">Phone</dt>
                  <dd className="mt-1 text-sm text-gray-900">{mediator.phone}</dd>
                </div>
              )}

              {mediator.yearsExperience && (
                <div>
                  <dt className="text-sm font-medium text-gray-500">Years of Experience</dt>
                  <dd className="mt-1 text-sm text-gray-900">{mediator.yearsExperience} years</dd>
                </div>
              )}

              {mediator.officeAddress && (
                <div>
                  <dt className="text-sm font-medium text-gray-500">Office Address</dt>
                  <dd className="mt-1 text-sm text-gray-900 whitespace-pre-line">{mediator.officeAddress}</dd>
                </div>
              )}

              {mediator.bio && (
                <div>
                  <dt className="text-sm font-medium text-gray-500">Bio</dt>
                  <dd className="mt-1 text-sm text-gray-900 whitespace-pre-line">{mediator.bio}</dd>
                </div>
              )}
            </dl>
          </div>

          {/* Specializations */}
          <div className="card">
            <h2 className="text-xl font-semibold mb-4">Specializations & Credentials</h2>
            
            <div className="space-y-4">
              <div>
                <h3 className="text-sm font-medium text-gray-500 mb-2">Specializations</h3>
                <div className="flex flex-wrap gap-2">
                  {mediator.specializations.length > 0 ? (
                    mediator.specializations.map((spec: string) => (
                      <span
                        key={spec}
                        className="px-3 py-1 bg-blue-100 text-blue-800 text-sm rounded-full"
                      >
                        {spec}
                      </span>
                    ))
                  ) : (
                    <span className="text-sm text-gray-500">None specified</span>
                  )}
                </div>
              </div>

              <div>
                <h3 className="text-sm font-medium text-gray-500 mb-2">Certifications</h3>
                <div className="flex flex-wrap gap-2">
                  {mediator.certifications.length > 0 ? (
                    mediator.certifications.map((cert: string) => (
                      <span
                        key={cert}
                        className="px-3 py-1 bg-green-100 text-green-800 text-sm rounded-full"
                      >
                        {cert}
                      </span>
                    ))
                  ) : (
                    <span className="text-sm text-gray-500">None specified</span>
                  )}
                </div>
              </div>

              {mediator.languages.length > 0 && (
                <div>
                  <h3 className="text-sm font-medium text-gray-500 mb-2">Languages</h3>
                  <div className="flex flex-wrap gap-2">
                    {mediator.languages.map((lang: string) => (
                      <span
                        key={lang}
                        className="px-3 py-1 bg-purple-100 text-purple-800 text-sm rounded-full"
                      >
                        {lang}
                      </span>
                    ))}
                  </div>
                </div>
              )}
            </div>
          </div>

          {/* Pricing */}
          {mediator.pricing && (
            <div className="card">
              <h2 className="text-xl font-semibold mb-4">Pricing</h2>
              
              <dl className="grid grid-cols-1 gap-4">
                {mediator.pricing.preMediationFee && (
                  <div>
                    <dt className="text-sm font-medium text-gray-500">Pre-Mediation Fee</dt>
                    <dd className="mt-1 text-sm text-gray-900">
                      ${mediator.pricing.preMediationFee.toNumber()}
                      {mediator.isGstRegistered && ' (inc GST)'}
                    </dd>
                  </div>
                )}

                <div>
                  <dt className="text-sm font-medium text-gray-500">Hourly Rate</dt>
                  <dd className="mt-1 text-sm text-gray-900">
                    ${mediator.pricing.hourlyRate.toNumber()}/hour
                    {mediator.pricing.hourlyRateBasis === 'PER_PARTY' && ' per party'}
                    {mediator.isGstRegistered && ' (inc GST)'}
                  </dd>
                </div>

                {mediator.pricing.travelIncludedKm && (
                  <div>
                    <dt className="text-sm font-medium text-gray-500">Travel Policy</dt>
                    <dd className="mt-1 text-sm text-gray-900">
                      {mediator.pricing.travelIncludedKm}km included, 
                      ${mediator.pricing.travelRatePerKm?.toNumber() || 0}/km thereafter
                    </dd>
                  </div>
                )}

                {mediator.pricing.accommodationPolicy && (
                  <div>
                    <dt className="text-sm font-medium text-gray-500">Accommodation</dt>
                    <dd className="mt-1 text-sm text-gray-900 whitespace-pre-line">
                      {mediator.pricing.accommodationPolicy}
                      {mediator.pricing.accommodationRate && 
                        ` - $${mediator.pricing.accommodationRate.toNumber()}`}
                    </dd>
                  </div>
                )}

                {mediator.pricing.extraCostsNotes && (
                  <div>
                    <dt className="text-sm font-medium text-gray-500">Additional Costs</dt>
                    <dd className="mt-1 text-sm text-gray-900 whitespace-pre-line">
                      {mediator.pricing.extraCostsNotes}
                    </dd>
                  </div>
                )}
              </dl>
            </div>
          )}
        </div>

        {/* Sidebar */}
        <div className="lg:col-span-1 space-y-6">
          {/* Status */}
          <div className="card">
            <h3 className="font-semibold text-gray-900 mb-4">Status</h3>
            <div className="space-y-3">
              <div>
                <div className="text-sm text-gray-500">Current Status</div>
                <div className="mt-1">
                  <span className="px-3 py-1 bg-yellow-100 text-yellow-800 text-sm rounded-full capitalize">
                    {mediator.status.toLowerCase().replace('_', ' ')}
                  </span>
                </div>
              </div>

              <div>
                <div className="text-sm text-gray-500">GST Registered</div>
                <div className="mt-1 text-sm text-gray-900">
                  {mediator.isGstRegistered ? 'Yes' : 'No'}
                  {mediator.gstNumber && (
                    <div className="text-xs text-gray-500 mt-1">
                      ABN: {mediator.gstNumber}
                    </div>
                  )}
                </div>
              </div>

              <div>
                <div className="text-sm text-gray-500">Practice Slug</div>
                <div className="mt-1 text-sm text-gray-900 font-mono">
                  /practices/{mediator.slug}
                </div>
              </div>

              <div>
                <div className="text-sm text-gray-500">Registered</div>
                <div className="mt-1 text-sm text-gray-900">
                  {new Date(mediator.user.createdAt).toLocaleDateString()}
                </div>
              </div>
            </div>
          </div>

          {/* Actions */}
          <MediatorReviewActions
            mediatorId={mediator.id}
            adminId={user.id}
            status={mediator.status}
          />
        </div>
      </div>
    </div>
  );
}
